extends PanelContainer

@export var mod_id: String = "CST1229-BALauncherModLoader";
@onready var mod_data: ModData = ModLoaderMod.get_mod_data(mod_id);

@onready var mod_settings = $ModSettings

@onready var mod_name = mod_data.manifest.name;

@onready var expand: Button = $VBoxContainer/Expand
@onready var toggle: TextureButton = $RightStuff/Toggle
@onready var type_icon: TextureRect = $LeftStuff/TypeIcon

@onready var settings_margin: MarginContainer = $VBoxContainer/SettingsMargin
@onready var settings_box: VBoxContainer = $VBoxContainer/SettingsMargin/SettingsBox
@onready var description: RichTextLabel = $VBoxContainer/SettingsMargin/SettingsBox/Description
@onready var metadata: Label = $VBoxContainer/SettingsMargin/SettingsBox/Metadata
@onready var load_errors: Label = $VBoxContainer/SettingsMargin/SettingsBox/LoadErrors
@onready var load_warnings: Label = $VBoxContainer/SettingsMargin/SettingsBox/LoadWarnings


@onready var select: AudioStreamPlayer = $Select

var expanded: bool = false:
	set(value):	
		expanded = value;
		settings_margin.visible = value;
		expand.icon = preload("res://mods-unpacked/CST1229-BALauncherModLoader/assets/down.png") if expanded else preload("res://mods-unpacked/CST1229-BALauncherModLoader/assets/right.png");
		if expanded:
			mod_settings.add_settings();
		else:
			mod_settings.clear_settings();

func _ready() -> void:
	mod_settings.mod_id = mod_id;
	mod_settings.mod_data = mod_data;
	
	var types = $"/root/ModLoader/CST1229-BALauncherModLoader".mod_types;
	type_icon.visible = false;
	if mod_data in types:
		type_icon.visible = true;
		match types[mod_data]:
			"mods", "user://mods":
				type_icon.texture = load("res://assets/ui/pixelicons/folder.png");
				type_icon.tooltip_text = "Launcher mod";
			"game-mods", "user://game-mods":
				type_icon.texture = load("res://mods-unpacked/CST1229-BALauncherModLoader/assets/barfy.png");
				type_icon.tooltip_text = "Game mod";
			_:
				type_icon.visible = false;
				
		
	
	# anti-softlock
	# (prevent users from disabling the mod menu through itself)
	toggle.disabled = mod_id == "CST1229-BALauncherModLoader";
	if toggle.disabled:
		toggle.modulate.a = 0.5;
	
	for tag: String in mod_data.manifest.tags:
		if tag.to_lower().begins_with("name:"):
			mod_name = tag.substr(5).strip_edges();
			break;
	
	expanded = false;
	expand.pressed.connect(func():
		expanded = !expanded;
		select.play();
	);
	toggle.button_pressed = mod_data.is_active;
	toggle.toggled.connect(func(enabled: bool):
		if enabled:
			ModLoaderUserProfile.enable_mod(mod_id);
		else:
			ModLoaderUserProfile.disable_mod(mod_id);
		toggled.emit(enabled);
	);
	
	description.bbcode_enabled = mod_data.manifest.description_rich != "";
	if description.bbcode_enabled:
		description.text = mod_data.manifest.description_rich;
	else:
		description.text = mod_data.manifest.description;
	description.meta_clicked.connect(func(url):
		var string := str(url);
		if string.begins_with("http://") or string.begins_with("https://"):
			OS.shell_open(string);
	);
	
	var metadata_strings := PackedStringArray();
	metadata_strings.append(mod_id);
	metadata_strings.append(mod_data.manifest.version_number);
	if !mod_data.manifest.authors.is_empty():
		metadata_strings.append("by " + ", ".join(mod_data.manifest.authors));
	if mod_data.manifest.website_url:
		metadata_strings.append(mod_data.manifest.website_url);
	metadata.text = "  -  ".join(metadata_strings);
	
	refresh_warnings();

func refresh_warnings():
	var errors := mod_data.load_errors.duplicate();
	errors.append_array(mod_data.manifest.validation_messages_error);
	for dependency in mod_data.manifest.dependencies:
		var data := ModLoaderMod.get_mod_data(dependency);
		if not data:
			errors.append("This mod has a missing dependency: " + dependency + ".");
		elif mod_data.is_active and not data.is_active:
			errors.append("This mod has a disabled dependency: " + dependency + ". Please enable it.");
	
	var warns := mod_data.load_warnings.duplicate();
	warns.append_array(mod_data.manifest.validation_messages_warning);
	if mod_data.is_active:
		for incompatibility in mod_data.manifest.incompatibilities:
			var data := ModLoaderMod.get_mod_data(incompatibility);
			if data and data.is_active:
				warns.append("This mod is incompatible with " + incompatibility + ".");
	
	load_errors.visible = !errors.is_empty();
	load_errors.text = "\n".join(errors);
	load_warnings.visible = !warns.is_empty();
	load_warnings.text = "\n".join(warns);
	expand.text = mod_name;
	if load_errors.visible or load_warnings.visible:
		expand.text = "(!)" + expand.text;
		if load_errors.visible:
			expand.modulate = load_errors.modulate;
		else:
			expand.modulate = load_warnings.modulate;
	else:
		expand.modulate = Color.WHITE;

func _on_mod_settings_config_changed() -> void:
	config_changed.emit();

signal toggled(on: bool);
signal config_changed();
