extends Node

var mod_id: String = "CST1229-BALauncherModLoader";
var mod_data: ModData;

@export var select: AudioStreamPlayer;
@export var refresh: AudioStreamPlayer;

@export var settings_box: Node;

var settings_nodes: Array[Node] = [];

func _ready():
	setup_label_settings();

var TITLE_LABEL_SETTINGS := LabelSettings.new();
var SETTING_LABEL_SETTINGS := LabelSettings.new();
var TEXTBOX_STYLEBOX := StyleBoxFlat.new();
var FOCUSED_TEXTBOX_STYLEBOX: StyleBoxFlat;
var DESCRIPTION_STYLEBOX := StyleBoxEmpty.new();
var SETTING_LABEL_STYLEBOX := StyleBoxEmpty.new();
const TITLE_FONT = preload("res://assets/ui/barfymono.png");
const DESCRIPTION_FONT = preload("res://mods-unpacked/CST1229-BALauncherModLoader/assets/nokiafc22.ttf");
func setup_label_settings():
	TITLE_LABEL_SETTINGS.font = TITLE_FONT;
	TITLE_LABEL_SETTINGS.font_size = 32;
	SETTING_LABEL_SETTINGS.font = DESCRIPTION_FONT;
	SETTING_LABEL_SETTINGS.font_size = 24;
	
	SETTING_LABEL_STYLEBOX.content_margin_left = 8;
	
	TEXTBOX_STYLEBOX.bg_color = Color("#000000a9");
	TEXTBOX_STYLEBOX.border_color = Color.WHITE;
	TEXTBOX_STYLEBOX.set_content_margin_all(16);
	TEXTBOX_STYLEBOX.set_border_width_all(2);
	FOCUSED_TEXTBOX_STYLEBOX = TEXTBOX_STYLEBOX.duplicate();
	FOCUSED_TEXTBOX_STYLEBOX.draw_center = false;
	FOCUSED_TEXTBOX_STYLEBOX.border_color = Color.GREEN;
	
	DESCRIPTION_STYLEBOX.content_margin_left = 44;

func clear_settings():
	for node in settings_nodes:
		if node is Control:
			(node as Control).release_focus();
		node.queue_free();
	settings_nodes = [];

func add_settings():
	clear_settings();
	if not mod_data.current_config or not mod_data.current_config.schema:
		return;
	var setting_schema = mod_data.current_config.schema;
	add_setting({data = mod_data.current_config.data}, &"data", setting_schema, true)

func add_setting(object: Dictionary, key: StringName, schema: Dictionary, is_root := false):
	# const properties can't be modified, no point in making them modifiable
	if schema.has(&"const"):
		return;
	
	var value = object.get(key, schema.get(&"default", ""));
	var title: String = schema.get(&"title", "");
	var format: String = schema.get(&"format", "");
	match schema.type:
		"object":
			if !is_root:
				add_setting_title(title);
			else:
				add_padding(48);
			for obj_key in schema.properties.keys():
				add_setting(value, obj_key, schema.properties[obj_key]);
			if !is_root:
				add_padding(16);
		"boolean":
			var checkbox: TextureButton = load("res://mods-unpacked/CST1229-BALauncherModLoader/scenes/Checkbox.tscn").instantiate();
			var box := make_base_setting(object, key, schema, func(default: bool):
				checkbox.set_pressed_no_signal(default);
				object[key] = default;
			);
			checkbox.size_flags_horizontal = Control.SIZE_SHRINK_END;
			checkbox.custom_minimum_size = Vector2(48, 48);
			checkbox.button_pressed = !!value;
			checkbox.toggled.connect(func(on: bool):
				object[key] = on;
				config_changed.emit();
			);
			box.add_child(checkbox);
		"string":
			if format == "color":

				var color := ColorPickerButton.new();
				var box := make_base_setting(object, key, schema, func(default: String):
					color.color = Color.from_string(default, Color.BLACK);
					object[key] = default;
				);
				color.custom_minimum_size.x = 72;
				color.custom_minimum_size.y = 48;
				color.size_flags_horizontal = Control.SIZE_SHRINK_END;
				color.flat = true;
				color.color = Color.from_string(str(value), Color.BLACK);
				color.color_changed.connect(func(new_color: Color):
					object[key] = "#" + new_color.to_html();
				);
				color.pressed.connect(func():
					select.play();
				);
				color.toggle_mode = false;
				box.add_child(color);
			elif format == "":
				var textbox := LineEdit.new();
				
				var pattern: RegEx = null;
				if "pattern" in schema:
					pattern = RegEx.new();
					if pattern.compile(schema.pattern) != OK:
						push_error("Could not compile pattern of " + key);
				var validate := func() -> bool:
					if pattern and pattern.search(textbox.text) == null:
						textbox.self_modulate = Color(1, 0.33, 0.33);
						return false;
					else:
						textbox.self_modulate = Color.WHITE;
						return true;
				
				var box := make_base_setting(object, key, schema, func(default: String):
					textbox.text = default;
					if validate.call():
						object[key] = textbox.text;
				);
				textbox.size_flags_horizontal = Control.SIZE_EXPAND_FILL;
				textbox.size_flags_stretch_ratio = 1.0;
				textbox.text = str(value);
				textbox.text_changed.connect(func(new_text: String):
					if validate.call():
						object[key] = new_text;
						config_changed.emit();
				);
				textbox.focus_entered.connect(func():
					select.play();
				);
				if "maximumLength" in schema:
					textbox.max_length = schema.maximumLength;
				
				
				textbox.add_theme_font_override(&"font", DESCRIPTION_FONT);
				textbox.add_theme_font_size_override(&"font_size", 16);
				textbox.add_theme_stylebox_override(&"normal", TEXTBOX_STYLEBOX);
				textbox.add_theme_stylebox_override(&"focus", FOCUSED_TEXTBOX_STYLEBOX);
				textbox.add_theme_constant_override(&"caret_width", 2);
				box.add_child(textbox);
			else:
				add_setting_title(key + ": Unsupported string format " + format);
		"number", "integer":
			var slider := SpinBox.new();
			var box := make_base_setting(object, key, schema, func(default: float):
				slider.set_value_no_signal(default);
				object[key] = default;
			);
			
			slider.size_flags_horizontal = Control.SIZE_EXPAND_FILL;
			slider.size_flags_stretch_ratio = 1.0;
			slider.add_theme_font_override(&"font", DESCRIPTION_FONT);
			slider.add_theme_font_size_override(&"font_size", 16);
			slider.add_theme_icon_override(&"updown", preload("res://mods-unpacked/CST1229-BALauncherModLoader/assets/updown.png"))
			slider.focus_entered.connect(func():
				select.play();
			);
			if "exclusiveMinimum" in schema:
				slider.min_value = schema.exclusiveMinimum + 0.0001;
			elif "minimum" in schema:
				slider.min_value = schema.minimum;
			if "exclusiveMaximum" in schema:
				slider.max_value = schema.exclusiveMaximum - 0.0001;
			elif "maximum" in schema:
				slider.max_value = schema.maximum;
			if "multipleOf" in schema:
				slider.step = schema.multipleOf;
				slider.custom_arrow_step = slider.step;
			else:
				slider.step = 0.0;
				slider.custom_arrow_step = 1.0;
			slider.rounded = schema.type == "integer";
			slider.value = value;
			slider.value_changed.connect(func(new_value: float):
				object[key] = new_value;
				config_changed.emit();
			);
			
			var line_edit := slider.get_line_edit();
			line_edit.add_theme_font_override(&"font", DESCRIPTION_FONT);
			line_edit.add_theme_font_size_override(&"font_size", 16);
			line_edit.add_theme_stylebox_override(&"normal", TEXTBOX_STYLEBOX);
			line_edit.add_theme_stylebox_override(&"focus", FOCUSED_TEXTBOX_STYLEBOX);
			line_edit.add_theme_constant_override(&"caret_width", 2);
			line_edit.focus_entered.connect(func():
				select.play();
			);
			
			box.add_child(slider);
		_:
			add_setting_title(key + ": Unsupported type " + schema.type);

func make_base_setting(object, key: String, schema: Dictionary, setter: Callable) -> HBoxContainer:
	var title: String = schema.get(&"title", key);
	var box := append_setting_node(HBoxContainer.new());
	
	if setter.is_valid():
		var default_button: TextureButton = load("res://ButtonHoverSilent.gd").new();
		default_button.custom_minimum_size = Vector2(32, 32);
		default_button.stretch_mode = TextureButton.STRETCH_KEEP_ASPECT_CENTERED;
		default_button.originalscale = 1;
		default_button.action_mode = BaseButton.ACTION_MODE_BUTTON_PRESS;
		default_button.texture_normal = preload("res://assets/ui/pixelicons/restart.png");
		default_button.size_flags_horizontal = Control.SIZE_SHRINK_BEGIN;
		default_button.size_flags_vertical = Control.SIZE_SHRINK_CENTER;
		default_button.pressed.connect(func():
			refresh.play();
			setter.call(schema.get(&"default", null));
			config_changed.emit();
		);
		box.add_child(default_button);
	setter.call(object.get(key, schema.get(&"default", "")));
	
	var label := Label.new();
	label.text = key if !title else title;
	label.label_settings = SETTING_LABEL_SETTINGS;
	label.size_flags_horizontal = Control.SIZE_EXPAND_FILL;
	label.vertical_alignment = VERTICAL_ALIGNMENT_CENTER;
	label.add_theme_stylebox_override(&"normal", SETTING_LABEL_STYLEBOX);
	label.autowrap_mode = TextServer.AUTOWRAP_WORD_SMART;
	box.add_child(label);
	
	if &"description" in schema:
		var description: RichTextLabel = append_setting_node(RichTextLabel.new(), false) as RichTextLabel;
		description.autowrap_mode = TextServer.AUTOWRAP_WORD_SMART;
		description.size_flags_horizontal = Control.SIZE_EXPAND_FILL;
		description.bbcode_enabled = false;
		description.fit_content = true;
		description.add_theme_stylebox_override(&"normal", DESCRIPTION_STYLEBOX);
		description.add_theme_font_override(&"normal_font", DESCRIPTION_FONT);
		description.add_theme_font_size_override(&"normal_font_size", 16);
		description.text = schema.description;
		description.meta_clicked.connect(func(url):
			var string := str(url);
			if string.begins_with("http://") or string.begins_with("https://"):
				OS.shell_open(string);
		);
	
	return box;

func add_setting_title(title := ""):
	if !title:
		return;
	add_padding(48);
	var label := Label.new();
	label.text = title;
	label.label_settings = TITLE_LABEL_SETTINGS;
	label.autowrap_mode = TextServer.AUTOWRAP_WORD_SMART;
	append_setting_node(label);
	add_padding(8);

func add_padding(padding: float):
	var control := Control.new();
	control.custom_minimum_size.y = padding;
	append_setting_node(control);
	
func append_setting_node(node: Node, pad: bool = true) -> Node:
	settings_nodes.append(node);
	settings_box.add_child(node);
	if pad && "custom_minimum_size" in node:
		node.custom_minimum_size.y = 24;
	return node;

signal config_changed();
